## -*-Tcl-*- (install)
 # ###################################################################
 #  Vince's Additions - an extension package for Alpha
 # 
 #  FILE: "superSearch.tcl"
 #                                    created: 1/10/97 {8:39:08 pm} 
 #                                last update: 1999-09-06T11:03:33Z 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501, USA
 #     www: <http://www.santafe.edu/~vince/>
 #  
 # Copyright (c) 1997-1999  Vince Darley, all rights reserved
 # 
 # See the file "license.terms" for information on usage and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 # 
 # Thanks for: 
 # 
 # Package modified on 5/7/99 by Dominique d'Humieres
 # E-mail: <dominiq@physique.ens.fr>
 # following ideas from Pierre Basso (3/07/98)
 # E-mail: <basso@lim.univ-mrs.fr>
 # 
 # Various notes:
 # 
 # The 'inselection' flag is ignored if you're doing a multiple-file
 # (batch or normal) search.
 # ###################################################################
 ##

# extension declaration
alpha::feature supersearch 0.7.9 global-only {
    alpha::package require AlphaTcl 7.2d1
    set supersearchOn 0
    array set supersearch {f 1 i 1 m 0 r 0 b 0 multi 0 \
      multifsets "" ismultifset 0 inselection 0 pattern "" wrap 1 ignore 0}
    namespace eval supersearch {}
    # Use this key combination to toggle the new search dialog on and off
    newPref binding supersearch::toggle "<U<O/F" global "" 1
    alpha::addToPreferencePage Packages supersearch::toggle supersearchOn
} {
    set supersearchOn 1
    supersearch::onoff
} {
    set supersearchOn 0
    supersearch::onoff
} maintainer {
    {Vince Darley} <vince@santafe.edu> <http://www.santafe.edu/~vince/>
} uninstall {this-file} help {
    Description:
	
    Togglable replacement for the 'Find' dialog box (shift-cmd-F is the
    default to toggle the dialog on/off)
	 
    Removes some deficiencies in Alpha's search dialog
    
    Allows single/multi-file regexp/batch/word-match search/replace/
    replace-all/in-selection from the same dialog box.
}

# Enter the number of rows of text to make room for in the 'Search For' and 
# 'Replace With' boxes in the search dialog
newPref var boxHeights 3 supersearch
# To clear the batch, multi-file and regexp flags when using shift-cmd-E 
# to enter a new search selection, click this box.||To leave all flags
# untouched when using shift-cmd-E, click this box.
newPref flag enterClearsBatch&MultiFlags 1 supersearch
# To turn Alpha's undo-memory off when doing 'replace all' (for added
# speed), click this box.||To remember all changes when doing 'replace all',
# click this box.
newPref flag undoOffForReplaceAll 1 supersearch
# To remember the current search fileset separately to Alpha's current
# fileset, click this box.||To synchronise the search fileset
# with Alpha's current fileset each time you open the search dialog, 
# click this box.
newPref flag separateSearchAndCurrentFileset 1 supersearch
# To set the 'in selection' flag automatically if there's a selection of at
# least 80 more characters than the length of the current search string,
# click this box.||To leave the 'in selection' flag in its original state,
# click this box.
newPref flag smartInSelectionFlag 1 supersearch
# To display tabs, newlines etc in the search dialog as '\t', \n' etc,
# click this box.||To leave the displayed search and replace strings
# unmodified, click this box.
newPref flag quoteFunnyChars 0 supersearch 
# Enter the regular expression defining the set of characters for which
# preceding or following spaces are ignored when the 'IgnoreSpaces' is on.
# If the 'regexp' flag is on, the set is replaced by '\r'.
# The set must not contain 'space' or 'tab' characters.
newPref var checkSpacesAround "\[^ a-zA-Z0-9----\t\]" supersearch

#menu -n supersearch -p menu::generalProc {search replace+find replaceAll replace}
#float -m supersearch -n "" -z super
namespace eval supersearch {}

proc supersearch::toggle {} {
    global supersearchOn
    set supersearchOn [expr {1 - $supersearchOn}]
    supersearch::onoff
    message "Supersearch switched [expr {$supersearchOn ? {on} : {off}}]"
}
if {[info tclversion] < 8.0} {
    proc supersearch::onoff {} {
	global supersearchOn  supersearch
	global buffersearch bufferreplace
	set buffersearch ""
	set bufferreplace ""
	set alpha_cmds {find findAgain replace replaceAll replace&FindAgain \
	  findInNextFile enterSearchString findAgainBackward searchString \
	  replaceString performSearch}
	if {$supersearchOn} {
	    if {[info commands alpha::_find] != ""} {return}
	    foreach p $alpha_cmds {
		if {![llength [info commands $p]]} { auto_load $p }
		rename $p alpha::_$p
		;proc $p {args} "eval supersearch::$p \$args"
	    }
	    # Tcl 7.5 has a bug in 'eval ... $args'
	    proc searchString {args} {
		global buffersearch supersearch
		switch -- [llength $args] {
		    0 {
			return $buffersearch
		    }
		    1 {
			set t [lindex $args 0]
			set supersearch(search) $t
			supersearch::parseSearch $t
			return [set buffersearch $t]
		    }
		    default {
			error "Too many args"
		    }
		}
	    }
	    proc replaceString {args} {
		global bufferreplace supersearch
		switch -- [llength $args] {
		    0 {
			return $bufferreplace
		    }
		    1 {
			set t [lindex $args 0]
			set supersearch(replace) $t
			return [set bufferreplace $t]
		    }
		    default {
			error "Too many args"
		    }
		}
	    }
	} else {
	    if {[info commands alpha::_find] == ""} {return}
	    foreach p $alpha_cmds {
		rename $p {}
		rename alpha::_$p $p
	    }
	}
    }
    proc supersearch::searchString {args} {
	global buffersearch supersearch
	switch -- [llength $args] {
	    0 {
		return $buffersearch
	    }
	    1 {
		set t [lindex $args 0]
		set supersearch(search) $t
		supersearch::parseSearch $t
		return [set buffersearch $t]
	    }
	    default {
		error "Too many args"
	    }
	}
    }
    
    proc supersearch::replaceString {args} {
	global bufferreplace supersearch
	switch -- [llength $args] {
	    0 {
		return $bufferreplace
	    }
	    1 {
		set t [lindex $args 0]
		set supersearch(replace) $t
		return [set bufferreplace $t]
	    }
	    default {
		error "Too many args"
	    }
	}
    }
    
} else {
    proc supersearch::onoff {} {
	global supersearchOn
	set alpha_cmds {find findAgain replace replaceAll replace&FindAgain \
	  findInNextFile enterSearchString findAgainBackward performSearch}
	if {$supersearchOn} {
	    if {[info commands ::alpha::_find] != ""} {return}
	    foreach p $alpha_cmds {
		if {![llength [info commands ::$p]]} { auto_load ::$p }
		rename ::$p ::alpha::_$p
		;proc ::$p {args} "eval ::supersearch::$p \$args"
	    }
	} else {
	    if {[info commands ::alpha::_find] == ""} {return}
	    foreach p $alpha_cmds {
		rename ::$p {}
		rename ::alpha::_$p ::$p
	    }
	}
    }
}

# set supersearch(start) 0
# set supersearch(end) 0

## 
 # -------------------------------------------------------------------------
 # 
 # "supersearch::performSearch" --
 # 
 #  Like 'search', but for use from Tcl code.  May have bugs still.
 # -------------------------------------------------------------------------
 ##
proc supersearch::performSearch {args} {
    global supersearch
    set opts(-b) 0
    set opts(-r) 0
    set opts(-f) 1
    set opts(-m) 0
    set opts(-i) 0
    getOpts {-f -r -m -b -i}
    if {[info exists opts(-s)]} {
	array set temp [array get supersearch]
    }
    set supersearch(multi) 0
    set supersearch(b) $opts(-b)
    set supersearch(r) $opts(-r)
    set supersearch(i) $opts(-i)
    set supersearch(m) $opts(-m)
    if {[info exists opts(-l)]} {
	set supersearch(inselection) 1
	if {$opts(-f)} {
	    set supersearch(start) [lindex $args 1]
	    set supersearch(end) $opts(-l)
	} else {
	    set supersearch(start) $opts(-l)
	    set supersearch(end) [lindex $args 1]
	}
    } else {
	set supersearch(inselection) 0
    }
    searchString [lindex $args 0]
    goto [lindex $args 1]
    supersearch::basicSearch $opts(-f)
    if {[info exists opts(-s)]} {
	array set supersearch [array get temp]
    }
}

proc supersearch::generalSearch {} {
    global supersearch
    if {$supersearch(b)} {
	supersearch::batchSearch
    } else {
	if {$supersearch(multi)} {
	    supersearch::getFiles
	    set supersearch(multiindex) -1
	    supersearch::findNextOkLocation
	    return
	} else {
	    if {$supersearch(inselection)} {
		goto $supersearch(start)
	    } else {
		goto [expr {$supersearch(f) ? [getPos] : [minPos]}]
	    }
	    supersearch::basicSearch
	}
    }
}

#  Basic searching  #
proc supersearch::findAgain {} {
    global supersearch
    if {$supersearch(b)} {set supersearch(b) 0}
    supersearch::basicSearch 1
}

proc supersearch::findAgainBackward {} {
    global supersearch
    if {$supersearch(b)} {
	dialog::errorAlert "You can't do backwards batch searches"
    } else {
	supersearch::basicSearch 0
    }
}

proc supersearch::basicSearch {{f 1} {rfrsh 0}} {
    global supersearch
    if {$supersearch(inselection) && !$supersearch(multi)} {
	set pos [getPos]
	set start $supersearch(start)
	set end $supersearch(end)
	set lstart [lindex [posToRowCol [lineStart $start]] 0]
	set lend [lindex [posToRowCol [nextLineStart $end]] 0]
	message "searching in selection between lines $lstart and $lend"
	if {[pos::compare $pos < $start] || [pos::compare $pos > $end]} {
	    goto [expr {$f ? $start : $end}]
	} 
    }
    set from $supersearch(reg)
    if {[catch {search -s -f $f -r 1 -i $supersearch(i) \
      -m $supersearch(m) -- $from [supersearch::searchStart $f]} p]} {
	supersearch::findInNextFile
    } else {
	if {$supersearch(inselection) && !$supersearch(multi)} {
	    if {$f} {
		if {[pos::compare [lindex $p 0] > $supersearch(end)]} {
		    supersearch::findNextOkLocation
		    return
		}
	    } else {
		if {[pos::compare [lindex $p 0] < $supersearch(start)]} {
		    supersearch::findNextOkLocation
		    return
		}
	    }
	}
	goto [lindex $p 0]
	getWinInfo wndw
	set wndwFrst $wndw(currline)
	set wndwDsp $wndw(linesdisp)
	set wndwln [lindex [posToRowCol [getPos]] 0]
	set wndwln [expr {4*(1-2*$f)*($wndwln-$wndwFrst)-(1-4*$f)*$wndwDsp}]
	if {$rfrsh||$wndwln < 0} {centerRedraw}
	eval select $p
    }
}
	
proc supersearch::replace {} {
    global supersearch
    set s [getPos] ; set e [selEnd]
    if {[pos::compare $s != $e]} {
	if {$supersearch(r)} {
	    if {$supersearch(i)} {
		set case "-nocase"
	    } else {
		set case "--"
	    }
	    regsub $case $supersearch(reg) [getText $s $e] [replaceString] rep
	    replaceText $s $e $rep
	} else {
	    replaceText $s $e [replaceString]
	}
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "supersearch::reg" --
 # 
 #  Take account of 'search' handling \t but regsub not handling it.
 # -------------------------------------------------------------------------
 ##
if {[info tclversion] < 8.1} {
    proc supersearch::reg {str} {
	regsub -all -- {(^|[^\\]|(\\\\)+)\\t} $str "\\1\t" str
	return "^$str\$"
    }
    proc supersearch::regPart {str} {
	regsub -all -- {(^|[^\\]|(\\\\)+)\\t} $str "\\1\t" str
	return "($str)"
    }
} else {
    proc supersearch::reg {str} {
	return "^$str\$"
    }
    proc supersearch::regPart {str} {
	return "($str)"
    }
}


proc supersearch::searchStart {{f 1}} {
    global supersearch
    if {[info exists supersearch(startfrom)]} {
	set p $supersearch(startfrom)
	unset supersearch(startfrom)
    } else {
	if {[pos::compare [set p [getPos]] != [selEnd]]} { 
	    if {$f} {
		set p [pos::math $p + 1]
	    } else {
		set p [pos::math $p - 1]
	    }
	}
    }
    set supersearch(laststart) $p
    return $p
}

proc supersearch::replace&FindAgain {} {
    supersearch::replace
    supersearch::findAgain
}

proc supersearch::enterSearchString {} {
    global supersearchmodeVars supersearch
    set t [getSelect]			
    searchString $t			
    set msg "Entered search '$t'"
    set supersearch(r) 0
    if {$supersearch(inselection)} {
	if {[getPos] >= $supersearch(start) && [getPos] <= $supersearch(end)} {
	    append msg ", still searching in previous selection."
	} else {
	    set supersearch(inselection) 0
	    append msg ", no longer searching in selection."
	}
    }
    if {$supersearchmodeVars(enterClearsBatch&MultiFlags)} {
	set supersearch(multi) 0
	set supersearch(b) 0
    }
    message [shorten $msg 80 45]
    supersearch::parseSearch
}

#  File switching  #
proc supersearch::nextFile {} {
    global supersearch
    set i 0
    while {$i < $supersearch(numfiles)} {
	if {[info exists supersearch(multiindex)] && $supersearch(multiindex) != ""} {
	    set f [lindex $supersearch(files) [incr supersearch(multiindex)]]
	    if {$f == ""} {
		set supersearch(files) ""
		unset supersearch(multiindex)
		return 0
	    } else {
		if {[supersearch::_isInFile $f [expr {$supersearch(numfiles) - $supersearch(multiindex) -1}]]} {
		    file::openQuietly $f
		    goto [minPos]
		    return 1
		} else {
		    incr i
		}
	    }
	}
	continue
    }
    return 0
}

proc supersearch::_isInFile {f {remaining ""}} {
    global supersearch
    set values 0
    
    set part $supersearch(firstLine)
    if {$supersearch(i)} {
	set case "-nocase"
	set scancase "-nocase"
    } else {
	set case "--"
	set scancase ""
    }
    set cid [scancontext create]
    set matches 0
    set lines {}
    set offset 0
    set fid 0
    eval scanmatch $scancase [list $cid $part {
	if {(!$supersearch(m) || [regexp -nocase "\\b$part\\b" $matchInfo(line)]) && \
	    !$matches} {
	    set offset $matchInfo(offset)
	    set lines $matchInfo(line)
	    set matches 1
	    set mlines $supersearch(rcFirst)
	    if {$offset == 0 && $mlines == 0 && \
	        [llength $supersearch(rcList)] != 0} {set matches  0 ; return $matches}
	    if {$mlines == 0} {return $matches}
	    if {$offset == 0 && $mlines == 2} {set matches  0 ; return $matches}
	    set from $supersearch(regTail)
	    if {$fid != 0} {close $fid}
	    set fid [open $f]
	    read $fid $offset
	    gets $fid lines
	    if {[eof $fid]} {set matches  0 ; return $matches}
	    while {$mlines > 0 } {
		if {$from == ""} {set mlines 0 ; return $matches}	 
		set mlines [regexp -indices "\\r" $from values]
		if {$mlines == 1} {
		    set rc [lindex $values 0]
		    set part [string range $from 0 [expr {$rc-1}]]
		    set from [string range $from [expr {$rc+1}] end]
		} else { 
		    set part $from
		}
		gets $fid lines
		if {[eof $fid]} {set matches  0 ; return $matches}
		if {$supersearch(ignore)} {
		    set lines [string trim $lines]
		}
		set part [supersearch::prepLine $part]
		set part "^$part"
		if {$mlines == 1} {set part "$part\$"}
		if {![regexp $case $part $lines]} {set matches  0 ; return $matches}
		continue	
	    }
	}
    }]
    
    if {$remaining == ""} {
	message "Searching [file tail $f]"
    } else {
	message "Searching ($remaining left) [file tail $f]"
    }
    
    if {![catch {open $f} fid]} {
	catch {scanfile $cid $fid}
	close $fid
    }
    
    scancontext delete $cid
    return $matches
}

proc supersearch::findNextOkLocation {} { supersearch::findInNextFile }
proc supersearch::findInNextFile {} {
    global supersearch
    if {$supersearch(multi)} {
	if {[supersearch::nextFile]} {
	    return [supersearch::basicSearch 1 1]
	}
	set str "Can't find '[searchString]', and there are no more files to search."
	message "[shorten $str 80 50]"
	set supersearch(numfiles) -1
    } else {
	if {$supersearch(inselection)} {
	    set str "No more instances of '[searchString]' in selected range."
	    message "[shorten $str 80 38]"
	} else {
	    if {$supersearch(wrap)} {
		if {$supersearch(b)} {
		    if {[pos::compare $supersearch(laststart) != [maxPos]]} {
			set supersearch(startfrom) [maxPos]
			message "Now searching backwards from end of document."
			return [supersearch::basicSearch 0]
		    }
		} else {
		    if {[pos::compare $supersearch(laststart) != [minPos]]} {
			set supersearch(startfrom) [minPos]
			message "Now searching forwards from start of document."
			return [supersearch::basicSearch 1]
		    }
		}
	    }
	    message "Can't find '[shorten [searchString] 65 30]'."
	}
    }
    beep
}

#  Batch  #
proc supersearch::batchSearch {} {
    global supersearch
    if {$supersearch(multi)} {
	supersearch::multiBatchSearch
    } else {
	supersearch::basicBatchSearch
    }
}

proc supersearch::basicBatchSearch {} {	
    global supersearch
    set from $supersearch(reg)
    if {$supersearch(m)} {
	set from "^.*\\b$from\\b.*$"
    } else {
	set from "^.*$from.*$"
    }
    set pos [expr {$supersearch(f) ? [getPos] : [minPos]}]
    set fileName [win::Current]
    set matches 0
    set lines {}
    while {![catch {search -s -f 1 -r 1 -i $supersearch(i) -- $from $pos} mtch]} {
	regsub -all "\t" [eval getText $mtch] "  " text
	append lines "\r" [format "Line %d: " \
	  [lindex [posToRowCol [lindex $mtch 0]] 0]] $text \
	  "\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t$fileName"
	set pos [lindex $mtch 1]
	incr matches
    }
    if {$matches} {
	grepsToWindow {* Matching Lines *} \
	  [format "%d matching lines (<cr> to go to match)\r-----" $matches] \
	  $lines "\r"
    } else {
	beep
	message "No matches found."
    }
}

proc supersearch::multiBatchSearch {} {
    global supersearch
    set from $supersearch(reg)
    if {$supersearch(m)} {
	set from "\\b$from\\b"
    }
    
    if {$supersearch(i)} {
	set case "-nocase"
    } else {
	set case "--"
    }
    
    supersearch::getFiles
    set supersearch(multiindex) -1
    set changes 0
    set i 0
    set values 0
    set matches 0
    set lines {}
    while {$i < $supersearch(numfiles)} {
	if {[info exists supersearch(multiindex)] && $supersearch(multiindex) != ""} {
	    set f [lindex $supersearch(files) [incr supersearch(multiindex)]]
	    if {$f == ""} {
		set supersearch(files) ""
		unset supersearch(multiindex)
		continue
	    } else {
		if {[supersearch::_isInFile $f [expr {$supersearch(numfiles) - $supersearch(multiindex) -1}]]} {
		    message "Looking at '[file tail $f]'"
		    if {![catch {open $f "r"} cid]} {
			if {[catch {read $cid} tmp]} {
			    alertnote "Opened $f, but couldn't read it!\
			      Will not be able to find matching lines."
			    close $cid
			} else {
			    close $cid
			    regsub -all "\n" $tmp "\r" tmp
			    set nl 1
			    set l [expr {40 - [string length [file tail $f]]}]
			    while {[regexp -indices $case "($from)" $tmp values]} {
				set m0 [lindex $values 0]
				set m1 [lindex $values 1]
				set beg [string range $tmp 0 [expr {$m0-1}]]
				set lrc [string last "\r" $beg]
				if {$lrc < 0} {
				    set m0 0
				} else {
				    set m0 [expr {$lrc+1}]
				} 
				set count [regsub -all "\r" $beg "\r" beg]
				incr nl $count
				set fol [string range $tmp $m1 end]
				set frc [string first "\r" $fol]
				if {$frc < 0} {
				    set m1 [expr {$m1+[string length "\r" $fol]}]
				} else {
				    set m1 [expr {$m1+$frc-1}]
				} 
				set mtch [string range $tmp $m0 $m1]
				set tmp [string range $tmp [expr {$m1+1}] end]
				regsub -all "\t" $mtch "  " text
				append lines "\r\"[file tail $f]\"; [format "%$l\s" ""]" [format "Line %d:\r" $nl] $mtch \
				  "\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t$f"
				incr matches
				set count [regsub -all "\r" $mtch "\r" beg]
				incr nl $count
			    }
			}
		    } else {
			alertnote "Couldn't open $f with read permission!\
			  Will not be able to find matching lines."
		    }
		}
		incr i
	    }
	    continue
	}
    }
	
    if {$matches} {
	grepsToWindow {* Matching Lines *} \
	  [format "%d matching lines (<cr> to go to match)\r-----" $matches] \
	  $lines "\r"
    } else {
	beep
	message "No matches found."
    }
}

#  Replace all  #
## 
 #  search  [options] <pattern> <pos> - 
 #   -f <num>      - go forward?
 #   -r <num>      - regular expression?
 #   -s            - save previous search string and search flags.
 #   -i <num>      - ignore case?
 #   -m <num>      - match words?
 #   -n            - failed search still returns TCL_OK, but null string.
 #   -l <limit>    - limit on how search goes.
 #   --            - next arg is the pattern.
 ##
proc supersearch::replaceAll {} {
    global supersearch
    if {$supersearch(multi)} {
	supersearch::multiReplaceAll
    } else {
	if {$supersearch(inselection)} {
	    goto $supersearch(start)
	} else {
	    goto [expr {$supersearch(f) ? [getPos] : [minPos]}]
	}
	supersearch::basicReplaceAll
    }
}

proc supersearch::basicReplaceAll {{f 1}} {
    global supersearch supersearchmodeVars undoOn
    set from $supersearch(reg)
    if {$supersearch(m)} {
	set from "\\b$from\\b"
    } 
    set savefrom [searchString]
    set to [replaceString]
    if {!$supersearch(r)} {
	set to [quote::Regsub $to]
	if {[catch {regsub -- $from "$savefrom" $to dummy} err]} {
	    alertnote "Regexp compilation problems: $err"
	    return
	}
	if {![llength {$dummy}]} {
	    alertnote "Regsub problem"
	    return
	}
    } else {
	if {[catch {regsub -- $from "$from" $to dummy} err]} {
	    alertnote "Regexp compilation problems: $err"
	    return
	}	
    }

    set pos [supersearch::searchStart]
    if {$supersearch(inselection) && !$supersearch(multi)} {
	set range [list $supersearch(start) $supersearch(end)]
    } else {
	if {$f} {
	    set range [list $pos [maxPos]]
	} else {
	    set range [list [minPos] $pos]
	}
    }
    if {$supersearch(i)} {
	set case "-nocase --"
    } else {
	set case "--"
    }
    if {[set count [eval regsub -all $case [list $from [eval getText $range] $to out]]]} {
	if {$supersearchmodeVars(undoOffForReplaceAll)} {
	    set oldUndo $undoOn ; set undoOn 0
	}	
	eval replaceText $range [list $out]
	if {$supersearchmodeVars(undoOffForReplaceAll)} {
	    set undoOn $oldUndo
	}
	message "Replacements: $count in given range: $range"
    } else {
	message "No replacements in given range: $range"
    }
}

proc supersearch::multiReplaceAll {} {
    global supersearch
    set from $supersearch(reg)
    set savefrom [searchString]
    set to [replaceString]
    if {!$supersearch(r)} {
	set to [quote::Regsub $to]
	if {[catch {regsub -- $from "$savefrom" $to dummy} err]} {
	    alertnote "Regexp compilation problems: $err"
	    return
	}
	if {![llength {$dummy}]} {
	    alertnote "Regsub problem"
	    return
	}
    } else {
	if {[catch {regsub -- $from "$from" $to dummy} err]} {
	    alertnote "Regexp compilation problems: $err"
	    return
	}	
    }
    supersearch::ensureAllWindowsSaved
    
    supersearch::getFiles
    set supersearch(multiindex) -1
    set changes 0
    set i 0
    while {$i < $supersearch(numfiles)} {
	if {[info exists supersearch(multiindex)] && $supersearch(multiindex) != ""} {
	    set f [lindex $supersearch(files) [incr supersearch(multiindex)]]
	    if {$f == ""} {
		set supersearch(files) ""
		unset supersearch(multiindex)
		continue
	    } else {
		if {[supersearch::_isInFile $f [expr {$supersearch(numfiles) - $supersearch(multiindex) -1}]]} {
		    message "Modifying ${f}"
		    if {![catch {open $f "r"} cid]} {
			set tmp [read $cid]
			close $cid
			regsub -all "\n" $tmp "\r" tmp
			if {$supersearch(i)} {
			    set inc [regsub -all -nocase -- $from $tmp $to tmp]
			} else {
			    set inc [regsub -all -- $from $tmp $to tmp]
			}
			
			if {$inc > 0} {
			    if {![catch {open $f "w+"} ocid]} {
				puts -nonewline $ocid $tmp
				close $ocid
				set tmp ""
				incr changes $inc
				set matches($f) 1
			    } else {
				alertnote "Couldn't open $f with write permission!  Changes will not take place."
			    }
			}
		    } else {
			alertnote "Couldn't open $f with read permission!  Changes will not take place."
		    }
		}
		incr i
	    }
	    continue
	}
    }

    eval file::revertThese [array names matches]
    message "Replaced $changes instances"
}



#  Search utilities  #
proc supersearch::getFiles {} {
    global supersearch
    set supersearch(files) ""
    foreach fset [supersearch::getfsets] {
	eval lappend supersearch(files) [getFileSet $fset]
    }
    set supersearch(numfiles) [llength $supersearch(files)]
    if {$supersearch(files) == ""} {
	dialog::errorAlert "To search multiple filesets, you must tell\
	  Alpha which filesets to search by clicking on the\
	  'multiple filesets' button.  Once you've done that \
	  once, you can then use the checkbox to toggle\
	  single/multi for all subsequent searches."
    }
}

proc supersearch::ensureAllWindowsSaved {} {
    global win::NumDirty
    if {${win::NumDirty}} {
	if {[buttonAlert "Save all windows?" "Yes" "Cancel"] != "Yes"} error
	saveAll
    }
}

proc supersearch::getfsets {} {
    global supersearch
    if {$supersearch(ismultifset)} {
	return $supersearch(multifsets)
    } else {
	return [list $supersearch(fileset)]
    }
}


#  Dialog box and helpers  #
proc supersearch::find {} {
    global supersearch searchPattern
    global supersearchmodeVars gfileSets currFileSet
    set multimenu $supersearch(multi)
    if {!$supersearchmodeVars(separateSearchAndCurrentFileset) \
      || ![info exists supersearch(fileset)] \
      || ($supersearch(fileset) == "")} {
	set supersearch(fileset) $currFileSet
    }
    set loop 0
    while 1 {
	if {[info tclversion] < 8.0 && $loop} {
	    # set search and replace strings
	    set newQuote $supersearchmodeVars(quoteFunnyChars)
	    if {($oldQuote != $newQuote)
		|| ($old != $str)} {
		    if {$oldQuote} {
			set str [quote::Undisplay $str]
		    }
		    if {[supersearch::updateStrings $str]} {return}
	    }
	}
	
	set y 5
	set args [dialog::text "Super search by Vince Darley" 50 y]
	incr y 5
	set yt $y
	if {$supersearchmodeVars(quoteFunnyChars)} {
	    eval lappend args [dialog::textedit "Search for:" \
	      [set supersearch(search) [quote::Display [searchString]]] \
	      10 y 27 $supersearchmodeVars(boxHeights)]
	    set y $yt
	    eval lappend args [dialog::textedit "Replace with:" \
	      [set supersearch(replace) [quote::Display [replaceString]]] \
	      300 y 27 $supersearchmodeVars(boxHeights)]
	} else {
	    eval lappend args [dialog::textedit "Search for:" \
	      [set supersearch(search) [searchString]] \
	      10 y 27 $supersearchmodeVars(boxHeights)]
	    set y $yt
	    eval lappend args [dialog::textedit "Replace with:" \
	      [set supersearch(replace) [replaceString]] \
	      300 y 27 $supersearchmodeVars(boxHeights)]
	}
	incr y 10
	set yr $y
	eval lappend args \
	  [dialog::checkbox "From current pos" $supersearch(f) 8 y] \
	  [dialog::checkbox "Ignore Case" $supersearch(i) 8 y] \
	  [dialog::checkbox "Regexp" $supersearch(r) 8 y] \
	  [dialog::checkbox "Batch" $supersearch(b) 8 y] 
	set my $y
	set y $yr
	eval lappend args \
	  [dialog::checkbox "Wrap " $supersearch(wrap) 162 y] \
	  [dialog::checkbox "Word Match" $supersearch(m) 120 y] \
	  [dialog::checkbox "Ignore spaces" $supersearch(ignore) 105 y]
	set y $my
	incr y 10
	eval lappend args [dialog::button "Search" 10 y "Don't Search" 90 y] \
	  [dialog::button "Replace All" 10 y "Cancel" 125 y]
	set ym $y
	set y 10 ; eval lappend args [dialog::button "prefs" 500 y]
	# single/multi files
	set y $yr
	set x 245
	set ftypes [list "Just one file" "Multiple Files" "Patterns"]
	eval lappend args \
	  [dialog::text "Options:" [expr {$x -20}] y] \
	  [dialog::menu $x y $ftypes [lindex $ftypes $multimenu]] 
	incr y 10
	set yr $y
	if {$supersearchmodeVars(smartInSelectionFlag)} {
	    if {([win::Current] != "") && ([string length [getSelect]] - \
	      [string length $supersearch(search)] > 80)} {
		set select 1
	    } else {
		set select 0
	    }
	} else {
	    set select $supersearch(inselection)
	}
	# If there's no selection move that checkbox offscreen.
	eval lappend args \
	  [list -n "Just one file"] \
	  [dialog::checkbox "In selection only" $select \
	  [expr {([win::Current] != "") && ([string length [getSelect]] > 0) ? $x : 3000}] y] \
	  [list -n "Multiple Files"]
	set y $yr
	eval lappend args \
	  [dialog::text "Select a single fileset" [expr {$x -20}] y]
	incr y -20
	eval lappend args \
	  [dialog::menu [expr {$x +140}] y \
	  [lsort -ignore [array names gfileSets]] \
	  $supersearch(fileset)]
	incr y 2
	if {[llength $supersearch(multifsets)]} {
	    eval lappend args \
	      [dialog::checkbox "or pick" $supersearch(ismultifset) [expr {$x -20}] y]
	} else {
	    # we don't want the check box, so we move it way offscreen.
	    eval lappend args \
	      [dialog::text "Or pick" [expr {$x -20}] y]
	    incr y -20
	    eval lappend args \
	      [dialog::checkbox "or pick" $supersearch(ismultifset) [expr {$x -1000}] y]
	}
	incr y -18
	eval lappend args \
	  [dialog::button "Multiple Filesets:" [expr {$x +55}] y] \
	  [dialog::text [join $supersearch(multifsets) ", "] $x y 50]
	set y [expr {$yr -32}]
	eval lappend args \
	  [dialog::button "New fileset" [expr {$x +145}] y \
	  "Dir scan" [expr {$x +255}] y]
	set y $yr
	eval lappend args \
	  [list -n "Patterns"] \
	  [dialog::button "Save this pattern" $x y] \
	  [dialog::menu $x y [array names searchPattern] $supersearch(pattern)] 
	incr y 4
	eval lappend args \
	  [dialog::button "Use pattern from menu" $x y]
	incr y 30
	if {[info tclversion] >= 8.0} {
	    lappend args -help [list "Enter the search string here" \
	      "Enter the replace string here" \
	      "To start the search at the current cursor position,\
	      click this box.||To search from the beginning of the file,\
	      click this box" \
	      "To ignore upper/lower case differences when searching,\
	      click this box.||To use exact, case-sensitive matches only,\
	      click this box" \
	      "To interpret the above search string as a regular expression,\
	      click this box.||To match the above search string literally,\
	      click this box." \
	      "To display all possible matches in a separate control window,\
	      click this box.||To highlight each match individually as it is\
	      found, click this box." \
	      "To continue searching from the beginning after reaching the end\
	      of the document, click this box.||To search only forwards from\
	      the given position, click this box." \
	      "To match only entire words,\
	      click this box.||To allow any sequence of characters which matches\
	      the above search string, click this box." \
	      "To ignore whitespace differences when searching,\
	      click this box.||To use exact, character and space matching only,\
	      click this box" \
	      "Click here to begin the search" \
	      "Click here to remember my settings, but not actually search" \
	      "Click here to replace all matches at once without my intervention" \
	      "Click here to discard any changes you've made to the settings." \
	      "Click here to edit the supersearch preferences." \
	      "More search facilities are accessible from this menu." \
	      "To search just in the current highlighted text,\
	      click this box||To search the entire text, click this box." \
	      "Select the fileset in which to search." \
	      "To search more than one fileset, use the 'Multiple filesets'\
	      button and click this box||To search just a single fileset,\
	      click this box." \
	      "Click here to select more than one fileset in which to search." \
	      "Click here to create a new fileset" \
	      "Click here to scan a particular directory." \
	      "Save the search and replace strings above for future use." \
	      "Select a pattern from this menu and then click the button below." \
	      "Use the pattern currently selected in this popup menu." \
	      ]
	}
	set res [eval dialog -w 580 -h $ym $args]
	set loop 1
	if {[info tclversion] < 8.0} {
	    set oldQuote $supersearchmodeVars(quoteFunnyChars)
	    if {$supersearchmodeVars(quoteFunnyChars)} {
		set old "\{[quote::Display [searchString]]\} \{[quote::Display [replaceString]]\}"
	    } else {
		set old "\{[searchString]\} \{[replaceString]\}"
	    }
	    # parse res, cor contains the flags and str contains the search and replace
	    # strings enclosed between braces: 
	    # cor="{} {} flags"
	    # str="{search} {replace}"
	    if {[supersearch::parse $res cor str]} {return}
	    set res $cor
	}
	
	if {[lindex $res 12]} {
	    return
	}
	if {[info tclversion] < 8.0} {
	    set i 2
	    foreach text {f i r b wrap m ignore} {
		set supersearch($text) [lindex $res $i]
		incr i
	    }
	    # set search and replace strings
# 	    if {$old != $str} {
# 		if {[supersearch::updateStrings $str]} {return}
# 	    }
	} else {
	    # turn on other flags if any
	    set i 0
	    foreach text {search replace f i r b wrap m ignore} {
		set supersearch($text) [lindex $res $i]
		incr i
	    }
	}
    
	if {$supersearchmodeVars(quoteFunnyChars)} {
	    searchString [quote::Undisplay $supersearch(search)]
	    replaceString [quote::Undisplay $supersearch(replace)]
	} else {
	    searchString $supersearch(search)
	    replaceString $supersearch(replace)
	}
# 	supersearch::parseSearch
	
	# get buttons
	set j $i
	foreach but {search dontSearch replaceAll cancel prefs} {
	    if {[lindex $res $j]} {
		set button $but
		break
	    }
	    incr j
	}
	
	incr i 5
	if {[lindex $res $i] == "Just one file"} {
	    set multimenu [set supersearch(multi) 0]
	    set supersearch(inselection) [lindex $res [incr i]]
	    if {$supersearch(inselection)} {
		if {[pos::compare [getPos] <= [selEnd]]} {
		    set supersearch(start) [getPos]
		    set supersearch(end) [selEnd]
		} else {
		    set supersearch(start) [selEnd]
		    set supersearch(end) [getPos]
		}
	    } 
	    incr i 6
	} elseif {[lindex $res $i] == "Multiple Files"} {
	    set multimenu [set supersearch(multi) 1]
	    set supersearch(inselection) [lindex $res [incr i]]
	    
	    # ignore multi-fset for the moment
	    set supersearch(fileset) [lindex $res [incr i]]
	    set supersearch(ismultifset) [lindex $res [incr i]]
	    
	    if {[lindex $res [incr i]]} {supersearch::multifset ; continue }
	    if {[lindex $res [incr i]]} {supersearch::newfset ; continue }
	    if {[lindex $res [incr i]]} {supersearch::dirscan ; continue }
	} else {
	    incr i 7
	    set multimenu 2
	    set supersearch(pattern) [lindex $res [expr {$i +1}]]
	    if {[lindex $res $i]} {
		supersearch::storePattern
		continue
	    }
	    if {[lindex $res [incr i 2]]} {supersearch::usePattern ; continue}
	    incr i
	}
	
	if {[info exists button] && $button == "prefs"} {
	    supersearch::prefs
	    unset button
	    continue
	}
	
	if {[info exists button]} {
	    if {$button == "search"} { 
		set button "generalSearch" 
	    }
	    if {[info tclversion] < 8.0} {
		# set search and replace strings
		if {($old != $str)} {
			if {$oldQuote} {
			    set str [quote::Undisplay $str]
			}
			if {[supersearch::updateStrings $str]} {return}
		}
	    }
	    supersearch::parseSearch
	    supersearch::$button
	    return
# 	    if {$button != "prefs"} {return}
	}
	
	set loop 0
	# big while loop
    }
}

if {[info tclversion] < 8.0} {
    
    ## 
     #	This proc search backward for the following pattern:
     #		
     #	'{s1} {s2} f f f f f f f f f f {s3} f {s4} f f f f f {s5} f '
     #	
     #		
     #	or '{s1} {s2} f f f f f f f f f f f {s3} f {s4} f f f f f {s5} f '
     #	it the flag 'Ignore Spaces' is used.
     #	
     #	The only assumptions are: s3, s4 and s5 are strings which do not
     #	contain an opening brace ("\{") and there are 10 binary flags
     #	between s2 and s3.  The proc returns in st the original string
     #	after replacing s1 and s2 by empty strings and "{s1} {s2}" in hd. 
     #	If s1 and s2 do not contain the string "\} \{" the proc
     #	supersearch::updateStrings sets silently the search and replace
     #	strings, otherwise it asks the user to set the boundary between the
     #	two strings.
     #		
     #	The use of prompt::var requires to use the useStatusBar and
     #	useStatusBar flags defined in this version as newPref flags.  This
     #	is not working properly.  The segmentation between search and
     #	replace string could be made easier by using the Pierre Basso idea
     #	of using marks.  His idea of inserting the replace string instead
     #	of replacing it could also be implemented.
     ##
    proc supersearch::parse {pat {st ""} {hd ""}} {
	if { $st != "" } {
	    upvar $st cor
	}
	if { $hd != "" } {
	    upvar $hd beg
	}
	set a [string last "\{" $pat]
	set new [string range $pat 0 [expr {$a - 1}]]
	set a [string last "\{" $new]
	set new [string range $new 0 [expr {$a - 1}]]
	set a [string last "\{" $new]
	set new [string range $new 0 [expr {$a - 26}]]
	set len [string length $new]
	set tail [string range $pat $len end]
	set cor "\{\} \{\}$tail"
	set beg $new
	return 0
    }
    
    proc supersearch::updateStrings {str} {
	global supersearch buffersearch bufferreplace
	set new $str
	set count [regsub -all "\} \{" $new "\}\{" withBullets]
	set len [string length $new]
	if {$count > 0} {
	    set a [string last "\} \{" $new]
	    if {$count >1} {
		set tmp $new
		set c $count
		while {$c > 0} {
		    set from [string range $new 1 [expr {$a - 1}]]
		    set to [string range $new [expr {$a + 3}] [expr {$len - 2}]]
		    set fl [string length $from]
		    set tl [string length $to]
		    if {$tl < 20} {
			set fl [expr {40 - $tl}]
		    } elseif {$fl < 20} {
			set tl [expr {40 - $fl}]
		    } else {
			set fl 20
			set tl 20
		    }  
		    set from [shorten $from $fl 8]
		    set to [shorten $to $tl 8]
		    lappend items "\"$from\" and \"$to\""
		    set tmp [string range $tmp 0 [expr {$a - 1}]]
		    set a [string last "\} \{" $tmp]
		    incr c -1
		}
		set item [listpick -p "Select the pair of strings you want." $items]
		set n [expr {$count - [lsearch -exact $items $item]}]
		set a [string last "\} \{" $new]
		if {$count < $n } {return 1}
		set tmp $new
		while {$count != $n} {
		    set tmp [string range $tmp 0 [expr {$a - 1}]]
		    set a [string last "\} \{" $tmp]
		    incr count -1
		}
	    }
	    set from [string range $new 1 [expr {$a - 1}]]
	    set buffersearch $from
	    set supersearch(search) $from
	    set to [string range $new [expr {$a + 3}] [expr {$len - 2}]]
	    set bufferreplace $to
	    set supersearch(replace) $to
	    return 0
	} else {
	    return 1
	}
    }
}

proc supersearch::prefs {} {
    catch {dialog::pkg_options supersearch}
}

proc supersearch::usePattern {} {
    global supersearch searchPattern
    set pats $searchPattern($supersearch(pattern))
    searchString [lindex $pats 0]
    replaceString [lindex $pats 1]
}

proc supersearch::storePattern {} {
    global supersearch searchPattern modifiedArrayElements
    if {[catch {set name [prompt "New pattern's name?" ""]}]} {
	return ""
    }
    lappend modifiedArrayElements [list $name searchPattern]
    set searchPattern($name) [list [searchString] [replaceString]]
    return $name
}

set {searchPattern(C++ to C Comments)} {{//(.*)} {/* \1 */}}

proc supersearch::dontSearch {} {}

proc supersearch::multifset {} {
    global supersearch gfileSets
    if {![catch {listpick -p "Select filesets" -l -L $supersearch(multifsets) \
      [lsort -ignore [array names gfileSets]]} res]} {
	if {[llength $res]} {
	    set supersearch(multifsets) $res
	    set supersearch(ismultifset) 1
	} else {
	    set supersearch(multifsets) ""
	    set supersearch(ismultifset) 0
	} 
    }
}

proc supersearch::dirscan {} {
    global gfileSets currFileSet gfileSetsType gDirScan
    global supersearch
    set supersearch(ismultifsets) 0
    if {[llength $supersearch(multifsets)] != 0} {
	set supersearch(multifsets) ""
    }
    
    set dir [get_directory -p "Scan which folder?"]
    if {![string length $dir]} return
    
    set filePat {*}
    set name [file tail $dir]
    
    set gfileSets($name) [file join $dir $filePat]
    set gDirScan($name) 1
    set gfileSetsType($name) "fromDirectory"
    set currFileSet $name
    updateCurrentFileset
    global supersearch currFileSet
    set supersearch(fileset) $currFileSet
}

proc supersearch::findNewDirectory {} {
    global gfileSets currFileSet gfileSetsType gDirScan
    
    set dir [get_directory -p "Scan which folder?"]
    if {![string length $dir]} return
    
    set filePat {*}
    set name [file tail $dir]
    
    set gfileSets($name) [file join $dir $filePat]
    set gDirScan($name) 1
    set gfileSetsType($name) "fromDirectory"
    set currFileSet $name
    updateCurrentFileset
    return $name
}

proc supersearch::newfset {} {
    if {[catch newFileset]} return
    global supersearch currFileSet
    set supersearch(fileset) $currFileSet
}

proc shorten {str {len 40} {el 0}} {
    if {[set sl [string length $str]] > $len} {
	set hl [expr {$len - $el - 2}]
	set str "[string range $str 0 $hl][string range $str [expr {$sl - $el}] end]"
    }
    return $str
}

proc supersearch::parseSearch {args} {
    global supersearch supersearchmodeVars
    switch -- [llength $args] {
	0 {
	    set from [searchString]
	}
	1 {
	    set from [lindex $args 0]
	}
	default {
	    error "Too many args"
	}
    }
    set values 0
    set rcList {}
    set tmp $from
    set mlines [regexp -indices "\\r" $tmp values]
    while {$mlines == 1} {
	set rc [lindex $values 0]
	lappend rcList $rc
	set tmp [string range $tmp [expr {$rc + 1}] end]
	set mlines [regexp -indices "\\r" $tmp values]
    }
    set pre ""
    set post ""
    set rcFirst 0
    set regTail ""
    if {[llength  $rcList] > 0} {
	set rc [lindex  $rcList 0]
	set post "\$"
	if {$supersearch(ignore)} {set post "\[ \t\]*$post"}
	if {$rc == 0} {
	    set pre "^"
	    if {$supersearch(ignore)} {set pre "$pre\[ \t\]*"}
	    if {[llength  $rcList] > 1} {
		set rc [lindex  $rcList 1]
		if {$rc == 0} {
		    set part ""
		} else { 
		    set part [string range $from 1 $rc]
		}
		set rcFirst 2
		set regTail [string range $from [expr {$rc + 2}] end]
	    } else { 
		set part [string range $from 1 end]
		set post ""
	    }
	} else { 
	    set part [string range $from 0 [expr {$rc - 1}]]
	    set rcFirst 1
	    set regTail [string range $from [expr {$rc + 1}] end]
	}
    } else { 
	set part $from
    }
    
    set part [supersearch::prepLine $part]
    set tmp $from
    if {!$supersearch(r)} {
	if {$supersearch(ignore)} {
	    set chr $supersearchmodeVars(checkSpacesAround)
	    set tmpf [regsub -all "\[ \t\]*($chr)\[ \t\]*" $tmp " \\1 " tmp]
	    set tmpf [regsub -- "^ " $tmp "" tmp]
	    set tmpf [regsub -- " $" $tmp "" tmp]
	    set tmp [quote::Regfind $tmp]
	    set tmpf [regsub -all "\[ \t\]+" $tmp "\[ \t\]*" tmp]
	} else {
	    set tmp [quote::Regfind $tmp]
        }
    } elseif {$supersearch(ignore)} {
	regsub -all {(\[([^\\]|\\[^r])*)\\r([^]]*\])} $tmp "\\1\\3" tmp
	regsub -all {(\[([^\\]|\\[^t])*)\\t([^]]*\])} $tmp "\\1\\3" tmp
	regsub -all {(\[([^\\]|\\[^n])*)\\n([^]]*\])} $tmp "\\1\\3" tmp
	regsub -all {(\[[^ ]*) ([^]]*\])} $tmp "\\1\\2" tmp
	regsub -all {(\[[^ ]*)	([^]]*\])} $tmp "\\1\\2" tmp
	set tmpf [regsub -all "\[ \t\]*\r\[ \t\]*" $tmp " \r " tmp]
	set tmpf [regsub -- "^ " $tmp "" tmp]
	set tmpf [regsub -- " $" $tmp "" tmp]
	set tmpf [regsub -all "\[ \t\]+" $tmp "\[ \t\]*" tmp]
	regsub -all {(\[([^]*|[^]*|[^]*|[^]*))([^]]*\])} $tmp "\\1\\r\\3" tmp	
	regsub -all {(\[([^]*|[^]*|[^]*|[^]*))([^]]*\])} $tmp "\\1\\t\\3" tmp	
	regsub -all {(\[([^]*|[^]*|[^]*|[^]*))([^]]*\])} $tmp "\\1\\n\\3" tmp	
	regsub -all {(\[([^]*|[^]*|[^]*))([^]]*\])} $tmp "\\1 \\3" tmp	
	regsub -all {(\[([^]*|[^]*|[^]*))([^]]*\])} $tmp "\\1	\\3" tmp	
    }
    set supersearch(rcList) $rcList
    set supersearch(rcFirst) $rcFirst
    set supersearch(firstLine) $pre$part$post
    set supersearch(regTail) $regTail
    set supersearch(reg) $tmp
}

proc supersearch::prepLine {str} {
    global supersearch supersearchmodeVars
    set tmp $str
    if {!$supersearch(r)} {
	set tmp [quote::Regfind $str]
	if {$supersearch(ignore)} {
	    set chr $supersearchmodeVars(checkSpacesAround)
	    set tmpf [regsub -all "\[ \t\]*($chr)\[ \t\]*" $str " \\1 " tmp]
	    set tmp "[string trim $tmp]"
	    set tmp [quote::Regfind $tmp]
	    set tmpf [regsub -all "\[ \t\]+" $tmp "\[ \t\]*" tmp]
	}
    } elseif {$supersearch(ignore)} {
	regsub -all {(\[[^ ]*) ([^]]*\])} $tmp "\\1\\2" tmp
	regsub -all {(\[[^	]*)	([^]]*\])} $tmp "\\1\\2" tmp
	set tmpf [regsub -all "\[ \t\]+" $tmp "\[ \t\]*" tmp]
	regsub -all {(\[([^]*|[^]*|[^]*|[^]*))([^]]*\])} $tmp "\\1	\\3" tmp	
	regsub -all {(\[([^]*|[^]*|[^]*|[^]*))([^]]*\])} $tmp "\\1 \\3" tmp	
    }
    return [supersearch::regPart $tmp] 
}

