/* -*-C++-*-
 * ###################################################################
 *  Cpptcl - connecting C++ with Tcl
 * 
 *  FILE: "tcl_strobj.h"
 *                                    created: 14/12/97 {6:48:13 pm} 
 *                                last update: 05/08/98 {10:00:04 AM} 
 *  Author: Vince Darley
 *  E-mail: <darley@fas.harvard.edu>
 *    mail: Division of Engineering and Applied Sciences, Harvard University
 *          Oxford Street, Cambridge MA 02138, USA
 *     www: <http://www.fas.harvard.edu/~darley/>
 *  
 * Copyright (c) 1997  Vince Darley
 * 
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *  Description: 
 * 
 *  History
 * 
 *  modified by  rev reason
 *  -------- --- --- -----------
 *  14/12/97 VMD 1.0 original
 * ###################################################################
 */

#ifndef _Cpptcl_tcl_strobj_
#define _Cpptcl_tcl_strobj_

#include <string.h>
#include "cpptclInt.h"

//@Section: Cpptcl library
//@Man: 
/** 
 *  "tcl_strobj" --
 *  
 *  Efficient wrapper to hold either a const char* or a Tcl_Obj*
 *  for safe-keeping.  If the latter, then it is freed when we go away.
 *  
 *  The main use of this is so that code can use constant strings "..."
 *  as well as variable-Tcl_Obj's.
 *  
 *  A subsidiary use is so aid some backwards compatibility with code
 *  which throws around 'const char*' everywhere.
 *  
 *  Note that Tcl considers the argument list it passes to commands
 *  to be owned by the Tcl library.  Hence items from that argument
 *  list should not be kept longer than the command is actually
 *  processed by your code.  If arguments are required to be kept,
 *  they must be duplicated first.
 *  
 *  For instance, two main ways in which the Cpptcl library will
 *  use the argument, besides comparing them with your objects'
 *  commands, are the following:
 *  
 *  (i) The name of a newly created object which is the argument
 *  following the creation command (e.g. Cpptest thisIsMyName),
 *  is duplicated with 'Tcl_DuplicateObj', and then the ref count
 *  is incremented.  This is because we're going to keep a copy
 *  of that name for the object's own purposes.
 *  
 *  (ii) As we read arguments off the given arg list, we keep 
 *  track of them inside the tcl_args class in the 'parsed_so_far'
 *  object instance.  This allows us to generate useful error
 *  messages afterwards, if we need to.  We also store a list of
 *  failed argument comparisons which is used for auto-completion
 *  and error messages.  Now these aren't copied at all.  We just
 *  put them in one of these 'tcl_strobj' instances, which increments
 *  their ref-count (so we know they're not changed behind our back).
 *  Just before we return control to Tcl, the tcl_args instance is
 *  deleted, and hence all those copied-pointers are released.
 *  
 *  If no error message/auto-completion takes place, in fact we 
 *  never even peek inside the pointers, so things are pretty fast.
 */
DLL_IMPORT_EXPORT
class tcl_strobj {
  private:
	/// anonymous union for char* or Tcl_Obj
	union {
		const char* str_s;
		mutable Tcl_Obj* str_o;
	};
	/// length of string if known, also tells us if it's a char* or an object.
	/** 
	 * if set >= -1 then it is a string (str_s), 
	 * and if == -2, it is an object
	 */
	mutable int length;
	///
	bool is_str(void) const { return (length != -2);}
	///
	bool is_obj(void) const { return (length == -2);}
	/// free any attachment to our contents.  Leaves object in UNSAFE state.
	void free(void) {if(is_obj() && (str_o != NULL)) { Tcl_DecrRefCount(str_o);}}
  public:
	/// Default constructor makes a null string.
	/** 
	 * The string you pass in here is not freed by us.  This class is designed
	 * to wrap around string literals "....", not general char*'s.
	 * Furthermore, WE DO NOT copy the string unless we are turned into
	 * a Tcl_Obj, so you should make sure the string doesn't change
	 * behind our back...
	 * 
	 * 'len' must be either -1 or the true length of the string.
	 * Other values will cause big run-time problems
	 */
	tcl_strobj(const char* stri=0,int len=-1):str_s(stri),length(len) {}
#if (TCL_MINOR_VERSION > 0)
	///
	tcl_strobj(const unsigned char* bytes,int len):length(len) {
		str_o = Tcl_NewByteArrayObj((unsigned char*)bytes,len);
		Tcl_IncrRefCount(str_o);
	}
#endif
	///
	tcl_strobj(const char* stri, int len, bool copy) {
		if(copy) {
			str_o = Tcl_NewStringObj((char*)stri,len); length = -2;
		} else {
			str_s = stri; length = len;
		}
	}
	/// Increments the ref-count so we know we have a valid representation.
	tcl_strobj(Tcl_Obj* obje):str_o(obje),length(-2) {
		if(str_o != 0) {
			Tcl_IncrRefCount(str_o);
		} else {
			// this is purely to cope with some compilers which call this
			// procedure instead of the char* one above, when passed a null
			// argument, and hence crash.
			str_s = 0; length = -1;
		}
	}
	/// a single character
	tcl_strobj(const char& c):str_s(&c),length(1) {}
	///
	tcl_strobj& operator=(const tcl_strobj& t) { 
		free();
		length = t.length;
		if(t.is_str()) {
			str_s = t.str_s;
		} else {
			str_o = t.str_o;
			if(str_o)
				Tcl_IncrRefCount(str_o);
		}
		return *this;
	}
	bool operator==(const tcl_strobj& o) { 
	    return !strcmp(operator const char*(),(const char*) o);
	}
	/// Only to be used for string literals, since we don't free the char*
	tcl_strobj& operator=(const char* stri) { 
		free(); str_s = stri; length = -1; return *this;
	}
	tcl_strobj& operator=(Tcl_Obj* obje) { 
		if(is_str() || obje != str_o) {
			free(); str_o = obje; 
			if(str_o)
				Tcl_IncrRefCount(str_o);
			length = -2;
		}
		return *this;
	}
	void clear(void) {
		if(is_str() || str_o != 0) {
			free(); str_o = 0; 
			length = -2;
		}
	}
	bool operator !=(Tcl_Obj* obje) {
		return (is_str() || obje != str_o ? true : false);
	}
	tcl_strobj(const tcl_strobj& t):length(t.length) {
		if(t.is_str()) {
			str_s = t.str_s;
		} else {
			str_o = t.str_o;
			Tcl_IncrRefCount(str_o);
		}
	}
	/// Is our string non-empty?
	operator bool (void) const { 
		return (is_str() ? str_s != NULL : str_o != NULL);
	}
	/// This is safe and efficient to use even if I am empty
	void append(const tcl_strobj&);
	/// This is safe and efficient to use even if I am empty
	void lappend(const tcl_strobj&);
	
	~tcl_strobj(void) {free();}
	/// If my string is a shared object, force my ownership
	/** 
	 * If my string is a char*, this call will do nothing, so I don't
	 * create new Tcl_Obj's unnecessarily.
	 * 
	 * Returns 'true' if I'm non-empty
	 */
	bool own(void) const;
	/// As above, but guarantees a valid Tcl_Obj.
	Tcl_Obj* ownobj(void) const;
	/// Will convert objects.  Don't call unless necessary
	operator const char* (void) const {
		if(is_str()) { 
			return str_s; 
		} else {
			register int len;
			return (str_o ? Tcl_GetStringFromObj(str_o, &len) : 0);
		}
	}
	/// Add my contents to the given object (which can be null)
	/**
	 * Note that if my contents are non-null, then this proc always returns
	 * a valid object, even if passed nothing.  However if my contents are
	 * null, this procedure returns immediately without examining 'o'
	 */
	Tcl_Obj* append_me_to(Tcl_Obj* &o) const;
	/// expensive if called foolishly
	operator Tcl_Obj* (void) const { 
		if(is_str()) ownobj(); 
		return str_o;
	}
	bool is_empty(void) const {return !(this->operator bool ());}
};

#endif

